<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AMP Generator - OLXTOTO</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', system-ui, sans-serif;
        }

        body {
            background: linear-gradient(135deg, #0a0a0a 0%, #1a1a1a 100%);
            color: #fff;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.05);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 20px 50px rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 2px solid rgba(43, 255, 0, 0.3);
        }

        .header h1 {
            font-size: 2.5rem;
            background: linear-gradient(90deg, #2bff00, #00ffde);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 10px;
        }

        .header p {
            color: #aaa;
            font-size: 1.1rem;
        }

        .form-container {
            display: grid;
            gap: 25px;
        }

        .form-group {
            position: relative;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #2bff00;
            font-weight: 600;
            font-size: 1.1rem;
        }

        .form-group input,
        .form-group textarea {
            width: 100%;
            padding: 15px;
            background: rgba(0, 0, 0, 0.3);
            border: 2px solid rgba(43, 255, 0, 0.2);
            border-radius: 10px;
            color: #fff;
            font-size: 1rem;
            transition: all 0.3s ease;
        }

        .form-group input:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #2bff00;
            box-shadow: 0 0 20px rgba(43, 255, 0, 0.2);
        }

        .form-group textarea {
            min-height: 120px;
            resize: vertical;
        }

        .char-count {
            position: absolute;
            right: 10px;
            bottom: 10px;
            color: #666;
            font-size: 0.9rem;
        }

        .preview-section {
            margin: 30px 0;
            padding: 25px;
            background: rgba(0, 0, 0, 0.3);
            border-radius: 15px;
            border: 1px solid rgba(43, 255, 0, 0.1);
        }

        .preview-section h3 {
            color: #2bff00;
            margin-bottom: 15px;
            font-size: 1.3rem;
        }

        .filename-preview {
            font-family: monospace;
            padding: 15px;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 8px;
            border-left: 4px solid #2bff00;
            margin: 10px 0;
            word-break: break-all;
        }

        .button-group {
            display: flex;
            gap: 20px;
            margin-top: 30px;
        }

        .btn {
            flex: 1;
            padding: 18px;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
            text-decoration: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .btn-primary {
            background: linear-gradient(90deg, #00ff22, #00ffde);
            color: #000;
        }

        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0, 255, 34, 0.3);
        }

        .btn-secondary {
            background: rgba(255, 255, 255, 0.1);
            color: #fff;
            border: 2px solid rgba(43, 255, 0, 0.3);
        }

        .btn-secondary:hover {
            background: rgba(43, 255, 0, 0.1);
            border-color: #2bff00;
        }

        .info-box {
            margin-top: 25px;
            padding: 20px;
            background: rgba(43, 255, 0, 0.05);
            border-radius: 12px;
            border-left: 4px solid #2bff00;
        }

        .info-box h4 {
            color: #2bff00;
            margin-bottom: 10px;
            font-size: 1.1rem;
        }

        .info-box ul {
            list-style: none;
            padding-left: 20px;
        }

        .info-box li {
            margin-bottom: 8px;
            color: #ccc;
            position: relative;
        }

        .info-box li:before {
            content: "•";
            color: #2bff00;
            position: absolute;
            left: -15px;
        }

        @keyframes pulse {
            0% { opacity: 0.6; }
            50% { opacity: 1; }
            100% { opacity: 0.6; }
        }

        .loading {
            animation: pulse 1.5s infinite;
        }

        @media (max-width: 768px) {
            .container {
                padding: 20px;
            }
            
            .button-group {
                flex-direction: column;
            }
            
            .header h1 {
                font-size: 2rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>AMP Generator</h1>
            <p>Generate file AMP untuk OLXTOTO dengan mudah</p>
        </div>

        <form id="ampForm" class="form-container">
            <div class="form-group">
                <label for="title">Judul Halaman ({{title}})</label>
                <input type="text" id="title" placeholder="Masukkan judul halaman..." required>
                <div class="char-count">0/60</div>
            </div>

            <div class="form-group">
                <label for="url">URL Halaman ({{url}})</label>
                <input type="url" id="url" placeholder="https://example.com/page" required>
            </div>

            <div class="form-group">
                <label for="description">Deskripsi ({{description}})</label>
                <textarea id="description" placeholder="Masukkan deskripsi halaman..." required></textarea>
                <div class="char-count">0/160</div>
            </div>

            <div class="form-group">
                <label for="url-gambar">URL Gambar ({{url-gambar}})</label>
                <input type="url" id="url-gambar" placeholder="https://example.com/image.jpg" required>
            </div>

            <div class="preview-section">
                <h3>Preview Nama File:</h3>
                <div id="filenamePreview" class="filename-preview">
                    Isi form untuk melihat preview nama file...
                </div>
                <p style="color: #aaa; font-size: 0.9rem; margin-top: 10px;">
                    File akan disimpan dengan format: amp-[domain].html
                </p>
            </div>

            <div class="info-box">
                <h4>Informasi Penting:</h4>
                <ul>
                    <li>Pastikan semua field telah diisi dengan benar</li>
                    <li>URL harus valid dan lengkap (termasuk https://)</li>
                    <li>Gambar harus memiliki rasio yang sesuai (1946x1300)</li>
                    <li>File akan digenerate dengan ekstensi .html</li>
                    <li>Nama file akan otomatis dibuat berdasarkan URL</li>
                </ul>
            </div>

            <div class="button-group">
                <button type="submit" class="btn btn-primary">
                    <span id="submitText">Generate File AMP</span>
                    <span id="loadingIcon" style="display: none;">🔄</span>
                </button>
                <button type="button" id="previewBtn" class="btn btn-secondary">
                    Preview Nama File
                </button>
            </div>
        </form>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.getElementById('ampForm');
            const titleInput = document.getElementById('title');
            const urlInput = document.getElementById('url');
            const descInput = document.getElementById('description');
            const imageInput = document.getElementById('url-gambar');
            const filenamePreview = document.getElementById('filenamePreview');
            const previewBtn = document.getElementById('previewBtn');
            const submitBtn = form.querySelector('button[type="submit"]');
            const submitText = document.getElementById('submitText');
            const loadingIcon = document.getElementById('loadingIcon');
            
            const titleCharCount = titleInput.parentElement.querySelector('.char-count');
            const descCharCount = descInput.parentElement.querySelector('.char-count');
            
            // Update character count
            titleInput.addEventListener('input', function() {
                titleCharCount.textContent = `${this.value.length}/60`;
            });
            
            descInput.addEventListener('input', function() {
                descCharCount.textContent = `${this.value.length}/160`;
            });
            
            // Preview filename function
            function updateFilenamePreview() {
                const url = urlInput.value.trim();
                if (url) {
                    try {
                        const urlObj = new URL(url);
                        let domain = urlObj.hostname.replace(/^www\./, '');
                        // Remove special characters and replace dots with hyphens
                        domain = domain.replace(/[^a-zA-Z0-9.-]/g, '-').replace(/\./g, '-');
                        // Remove consecutive hyphens
                        domain = domain.replace(/--+/g, '-');
                        // Remove leading/trailing hyphens
                        domain = domain.replace(/^-+|-+$/g, '');
                        
                        const filename = `amp-${domain}.html`;
                        filenamePreview.innerHTML = `
                            <strong>Nama File:</strong> ${filename}
                            <br><br>
                            <strong>URL Asal:</strong> ${url}
                            <br>
                            <strong>Domain:</strong> ${urlObj.hostname}
                        `;
                    } catch (e) {
                        filenamePreview.textContent = "URL tidak valid. Pastikan format URL benar.";
                    }
                } else {
                    filenamePreview.textContent = "Isi form untuk melihat preview nama file...";
                }
            }
            
            // Update preview on URL change
            urlInput.addEventListener('input', updateFilenamePreview);
            previewBtn.addEventListener('click', updateFilenamePreview);
            
            // Form submission
            form.addEventListener('submit', async function(e) {
                e.preventDefault();
                
                // Validate all fields
                if (!titleInput.value.trim() || !urlInput.value.trim() || 
                    !descInput.value.trim() || !imageInput.value.trim()) {
                    alert('Harap isi semua field!');
                    return;
                }
                
                try {
                    // Validate URL
                    new URL(urlInput.value);
                    new URL(imageInput.value);
                } catch (error) {
                    alert('Format URL tidak valid. Pastikan URL diawali dengan https://');
                    return;
                }
                
                // Show loading state
                submitText.textContent = 'Processing...';
                loadingIcon.style.display = 'inline';
                submitBtn.disabled = true;
                
                try {
                    // Get the template content
                    const response = await fetch('place-amp.html');
                    const template = await response.text();
                    
                    // Replace placeholders
                    const content = template
                        .replace(/{{title}}/g, titleInput.value.trim())
                        .replace(/{{url}}/g, urlInput.value.trim())
                        .replace(/{{description}}/g, descInput.value.trim())
                        .replace(/{{url-gambar}}/g, imageInput.value.trim());
                    
                    // Generate filename
                    const urlObj = new URL(urlInput.value.trim());
                    let domain = urlObj.hostname.replace(/^www\./, '');
                    domain = domain.replace(/[^a-zA-Z0-9.-]/g, '-').replace(/\./g, '-');
                    domain = domain.replace(/--+/g, '-').replace(/^-+|-+$/g, '');
                    const filename = `amp-${domain}.html`;
                    
                    // Create download
                    const blob = new Blob([content], { type: 'text/html' });
                    const downloadUrl = URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = downloadUrl;
                    a.download = filename;
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    URL.revokeObjectURL(downloadUrl);
                    
                    // Success message
                    alert(`File berhasil digenerate: ${filename}`);
                    
                } catch (error) {
                    console.error('Error:', error);
                    alert('Terjadi kesalahan saat generate file. Pastikan template tersedia.');
                } finally {
                    // Reset loading state
                    submitText.textContent = 'Generate File AMP';
                    loadingIcon.style.display = 'none';
                    submitBtn.disabled = false;
                }
            });
            
            // Auto-preview on load
            updateFilenamePreview();
        });
    </script>
</body>
</html>