<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OLXTOTO Landing Page Generator</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&family=Inter:wght@400;500;600&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #f8c300;
            --primary-dark: #e6b400;
            --secondary: #242424;
            --dark: #121212;
            --light: #f8f9fa;
            --success: #07a312;
            --danger: #e11d1d;
            --border: #2d2d2d;
            --card-bg: rgba(26, 26, 26, 0.9);
        }

        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #0a0a0a 0%, #1a1a1a 100%);
            color: var(--light);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
            padding: 30px 0;
            background: var(--card-bg);
            border-radius: 20px;
            border: 1px solid var(--primary);
            position: relative;
            overflow: hidden;
        }

        .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--success), var(--primary));
        }

        .header h1 {
            font-family: 'Poppins', sans-serif;
            font-size: 2.8rem;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), #fffb00);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 10px;
        }

        .header p {
            color: #aaa;
            font-size: 1.1rem;
            max-width: 600px;
            margin: 0 auto;
        }

        .main-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
            margin-bottom: 40px;
        }

        @media (max-width: 992px) {
            .main-grid {
                grid-template-columns: 1fr;
            }
        }

        .form-section, .preview-section {
            background: var(--card-bg);
            border-radius: 20px;
            padding: 30px;
            border: 1px solid var(--border);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
            backdrop-filter: blur(10px);
        }

        .section-title {
            font-family: 'Poppins', sans-serif;
            font-size: 1.5rem;
            color: var(--primary);
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--border);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .section-title i {
            font-size: 1.8rem;
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        .form-group {
            margin-bottom: 25px;
        }

        .form-group.full-width {
            grid-column: 1 / -1;
        }

        label {
            display: block;
            margin-bottom: 8px;
            color: var(--primary);
            font-weight: 600;
            font-size: 0.95rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .input-group {
            position: relative;
        }

        .input-group i {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--primary);
            z-index: 1;
        }

        input, textarea, select {
            width: 100%;
            padding: 15px 15px 15px 45px;
            background: rgba(40, 40, 40, 0.8);
            border: 2px solid var(--border);
            border-radius: 12px;
            color: var(--light);
            font-size: 1rem;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }

        input:focus, textarea:focus, select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(248, 195, 0, 0.2);
            background: rgba(50, 50, 50, 0.9);
        }

        textarea {
            min-height: 120px;
            resize: vertical;
            padding: 15px;
        }

        .input-group input {
            padding-left: 45px;
        }

        .faq-section {
            background: rgba(40, 40, 40, 0.6);
            border-radius: 12px;
            padding: 20px;
            margin-top: 10px;
        }

        .faq-item {
            margin-bottom: 20px;
            padding: 15px;
            background: rgba(30, 30, 30, 0.8);
            border-radius: 10px;
            border-left: 4px solid var(--primary);
        }

        .faq-item label {
            color: #fff;
            text-transform: none;
            font-size: 1rem;
        }

        .preview-box {
            background: rgba(20, 20, 20, 0.9);
            border-radius: 12px;
            padding: 25px;
            min-height: 400px;
            max-height: 500px;
            overflow-y: auto;
            border: 2px dashed var(--border);
        }

        .preview-item {
            background: rgba(40, 40, 40, 0.7);
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 10px;
            border-left: 4px solid var(--success);
        }

        .preview-item h4 {
            color: var(--primary);
            margin-bottom: 8px;
            font-size: 1.1rem;
        }

        .preview-item p {
            color: #ccc;
            font-size: 0.95rem;
            line-height: 1.5;
        }

        .preview-item .url {
            color: #4dabf7;
            word-break: break-all;
            font-size: 0.9rem;
        }

        .action-buttons {
            display: flex;
            gap: 20px;
            margin-top: 40px;
            padding-top: 30px;
            border-top: 2px solid var(--border);
        }

        .btn {
            padding: 18px 35px;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            font-family: 'Poppins', sans-serif;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .btn-primary {
            flex: 2;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: #000;
        }

        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(248, 195, 0, 0.3);
        }

        .btn-secondary {
            flex: 1;
            background: var(--secondary);
            color: var(--light);
            border: 2px solid var(--border);
        }

        .btn-secondary:hover {
            background: #333;
            border-color: var(--primary);
        }

        .btn-danger {
            background: var(--danger);
            color: white;
            padding: 12px 25px;
            font-size: 1rem;
        }

        .btn-danger:hover {
            background: #c21818;
            transform: translateY(-2px);
        }

        .notification {
            position: fixed;
            top: 30px;
            right: 30px;
            padding: 20px 30px;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            z-index: 1000;
            animation: slideIn 0.3s ease;
            display: flex;
            align-items: center;
            gap: 15px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
        }

        .success {
            background: linear-gradient(135deg, var(--success), #0b8a14);
            border-left: 5px solid #0cfc1c;
        }

        .error {
            background: linear-gradient(135deg, var(--danger), #c21818);
            border-left: 5px solid #ff5252;
        }

        @keyframes slideIn {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        .file-list {
            margin-top: 30px;
            background: var(--card-bg);
            border-radius: 20px;
            padding: 30px;
        }

        .file-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .file-card {
            background: rgba(40, 40, 40, 0.8);
            border-radius: 12px;
            padding: 20px;
            border: 1px solid var(--border);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .file-card:hover {
            transform: translateY(-5px);
            border-color: var(--primary);
            box-shadow: 0 10px 20px rgba(248, 195, 0, 0.1);
        }

        .file-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 5px;
            height: 100%;
            background: linear-gradient(180deg, var(--primary), var(--success));
        }

        .file-name {
            color: var(--primary);
            font-weight: 600;
            font-size: 1.1rem;
            margin-bottom: 10px;
            word-break: break-all;
        }

        .file-date {
            color: #888;
            font-size: 0.9rem;
            margin-bottom: 15px;
        }

        .file-actions {
            display: flex;
            gap: 10px;
        }

        .file-actions a {
            padding: 8px 15px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            color: var(--light);
            text-decoration: none;
            font-size: 0.9rem;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .file-actions a:hover {
            background: var(--primary);
            color: #000;
        }

        .required::after {
            content: ' *';
            color: var(--danger);
        }

        .char-count {
            font-size: 0.85rem;
            color: #888;
            text-align: right;
            margin-top: 5px;
        }

        .char-count.warning {
            color: #ff9800;
        }

        .char-count.danger {
            color: var(--danger);
        }
    </style>
</head>
<body>
    <?php
    session_start();
    
    // Fungsi untuk sanitasi input
    function sanitize($data) {
        return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
    }
    
    // Fungsi untuk generate nama file dari URL
    function generateFilename($url) {
        $url = str_replace(['https://', 'http://', 'www.'], '', $url);
        $url = str_replace('/', '_', $url);
        $url = preg_replace('/[^a-zA-Z0-9_\-]/', '', $url);
        $url = strtolower($url);
        return $url . '.html';
    }
    
    // Proses form jika disubmit
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Ambil data dari form
        $data = [
            'title' => sanitize($_POST['title'] ?? ''),
            'url' => sanitize($_POST['url'] ?? ''),
            'description' => sanitize($_POST['description'] ?? ''),
            'amphtml' => sanitize($_POST['amphtml'] ?? ''),
            'url_gambar' => sanitize($_POST['url_gambar'] ?? ''),
            'no_hp' => sanitize($_POST['no_hp'] ?? '')
        ];
        
        // Ambil data FAQ
        for ($i = 1; $i <= 5; $i++) {
            $data["question_$i"] = sanitize($_POST["question_$i"] ?? '');
            $data["answer_$i"] = sanitize($_POST["answer_$i"] ?? '');
        }
        
        // Validasi data
        $errors = [];
        if (empty($data['title'])) $errors[] = 'Title harus diisi';
        if (empty($data['url'])) $errors[] = 'URL harus diisi';
        if (empty($data['url_gambar'])) $errors[] = 'URL Gambar harus diisi';
        
        if (empty($errors)) {
            try {
                // Baca template
                $template = file_get_contents('place.html');
                
                // Ganti semua placeholder
                $replacements = [
                    '{{title}}' => $data['title'],
                    '{{url}}' => $data['url'],
                    '{{description}}' => $data['description'],
                    '{{amphtml}}' => $data['amphtml'],
                    '{{url-gambar}}' => $data['url_gambar'],
                    '{{no-hp}}' => $data['no_hp']
                ];
                
                for ($i = 1; $i <= 5; $i++) {
                    $replacements["{{Question-$i}}"] = $data["question_$i"];
                    $replacements["{{Answer-$i}}"] = $data["answer_$i"];
                }
                
                $content = str_replace(array_keys($replacements), array_values($replacements), $template);
                
                // Generate nama file
                $filename = generateFilename($data['url']);
                
                // Simpan file
                file_put_contents($filename, $content);
                
                // Simpan data ke session untuk preview
                $_SESSION['last_generated'] = [
                    'filename' => $filename,
                    'data' => $data,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                
                $success = "Landing page berhasil dibuat: <strong>$filename</strong>";
            } catch (Exception $e) {
                $errors[] = "Error: " . $e->getMessage();
            }
        }
    }
    
    // Ambil daftar file HTML yang sudah dibuat
    $html_files = glob('*.html');
    usort($html_files, function($a, $b) {
        return filemtime($b) - filemtime($a);
    });
    ?>
    
    <div class="container">
        <!-- Header -->
        <div class="header">
            <h1><i class="fas fa-bolt"></i> OLXTOTO Landing Page Generator</h1>
            <p>Buat landing page OLXTOTO secara otomatis dengan mengisi form di bawah ini</p>
        </div>
        
        <!-- Notifikasi -->
        <?php if (isset($success)): ?>
        <div class="notification success">
            <i class="fas fa-check-circle fa-2x"></i>
            <div>
                <strong>Sukses!</strong><br>
                <?php echo $success; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($errors)): ?>
        <div class="notification error">
            <i class="fas fa-exclamation-triangle fa-2x"></i>
            <div>
                <strong>Error:</strong><br>
                <?php echo implode('<br>', $errors); ?>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Grid Utama -->
        <div class="main-grid">
            <!-- Form Section -->
            <div class="form-section">
                <h2 class="section-title"><i class="fas fa-edit"></i> Input Data Landing Page</h2>
                
                <form method="POST" action="" id="landingForm">
                    <div class="form-grid">
                        <!-- Basic Information -->
                        <div class="form-group full-width">
                            <label class="required">Judul Halaman</label>
                            <div class="input-group">
                                <i class="fas fa-heading"></i>
                                <input type="text" name="title" id="title" required 
                                       placeholder="Contoh: OLXTOTO Slot Gacor Terpercaya 2024"
                                       maxlength="100"
                                       value="<?php echo $_POST['title'] ?? ''; ?>">
                            </div>
                            <div class="char-count" id="titleCount">0/100</div>
                        </div>
                        
                        <div class="form-group">
                            <label class="required">URL Utama</label>
                            <div class="input-group">
                                <i class="fas fa-link"></i>
                                <input type="url" name="url" id="url" required 
                                       placeholder="https://OLXTOTO.com/slot-gacor"
                                       value="<?php echo $_POST['url'] ?? ''; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>URL AMP</label>
                            <div class="input-group">
                                <i class="fas fa-mobile-alt"></i>
                                <input type="url" name="amphtml" id="amphtml" 
                                       placeholder="https://OLXTOTO.com/amp/slot-gacor"
                                       value="<?php echo $_POST['amphtml'] ?? ''; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label class="required">URL Gambar</label>
                            <div class="input-group">
                                <i class="fas fa-image"></i>
                                <input type="url" name="url_gambar" id="url_gambar" required 
                                       placeholder="https://OLXTOTO.com/images/banner.jpg"
                                       value="<?php echo $_POST['url_gambar'] ?? ''; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Nomor HP</label>
                            <div class="input-group">
                                <i class="fas fa-phone"></i>
                                <input type="tel" name="no_hp" id="no_hp" 
                                       placeholder="+6281234567890"
                                       value="<?php echo $_POST['no_hp'] ?? ''; ?>">
                            </div>
                        </div>
                        
                        <div class="form-group full-width">
                            <label>Deskripsi</label>
                            <textarea name="description" id="description" 
                                      placeholder="Deskripsi lengkap tentang OLXTOTO slot..."
                                      maxlength="500"><?php echo $_POST['description'] ?? ''; ?></textarea>
                            <div class="char-count" id="descCount">0/500</div>
                        </div>
                        
                        <!-- FAQ Section -->
                        <div class="form-group full-width">
                            <h3 class="section-title" style="font-size: 1.2rem; margin-top: 30px;">
                                <i class="fas fa-question-circle"></i> FAQ Section
                            </h3>
                            
                            <div class="faq-section">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                <div class="faq-item">
                                    <label>Pertanyaan <?php echo $i; ?></label>
                                    <input type="text" name="question_<?php echo $i; ?>" 
                                           placeholder="Contoh: Apa itu OLXTOTO?"
                                           value="<?php echo $_POST["question_$i"] ?? ''; ?>">
                                    
                                    <label style="margin-top: 10px;">Jawaban <?php echo $i; ?></label>
                                    <textarea name="answer_<?php echo $i; ?>" 
                                              placeholder="Jawaban lengkap..."
                                              rows="3"><?php echo $_POST["answer_$i"] ?? ''; ?></textarea>
                                </div>
                                <?php endfor; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-magic"></i> Generate Landing Page
                        </button>
                        <button type="reset" class="btn btn-secondary" onclick="resetForm()">
                            <i class="fas fa-redo"></i> Reset Form
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Preview Section -->
            <div class="preview-section">
                <h2 class="section-title"><i class="fas fa-eye"></i> Preview Data</h2>
                
                <div class="preview-box" id="previewBox">
                    <?php if (isset($_SESSION['last_generated'])): ?>
                    <div class="preview-item">
                        <h4><i class="fas fa-file-code"></i> File Terakhir Dibuat:</h4>
                        <p class="url"><?php echo $_SESSION['last_generated']['filename']; ?></p>
                        <small><?php echo $_SESSION['last_generated']['timestamp']; ?></small>
                    </div>
                    <?php endif; ?>
                    
                    <div class="preview-item">
                        <h4>Title:</h4>
                        <p id="previewTitle"><?php echo $_SESSION['last_generated']['data']['title'] ?? 'Belum diisi'; ?></p>
                    </div>
                    
                    <div class="preview-item">
                        <h4>URL:</h4>
                        <p class="url" id="previewUrl"><?php echo $_SESSION['last_generated']['data']['url'] ?? 'Belum diisi'; ?></p>
                    </div>
                    
                    <div class="preview-item">
                        <h4>Deskripsi:</h4>
                        <p id="previewDesc"><?php 
                            $desc = $_SESSION['last_generated']['data']['description'] ?? 'Belum diisi';
                            echo strlen($desc) > 100 ? substr($desc, 0, 100) . '...' : $desc;
                        ?></p>
                    </div>
                    
                    <div class="preview-item">
                        <h4>Gambar:</h4>
                        <p class="url" id="previewImage"><?php echo $_SESSION['last_generated']['data']['url_gambar'] ?? 'Belum diisi'; ?></p>
                    </div>
                    
                    <div class="preview-item">
                        <h4>Nomor HP:</h4>
                        <p id="previewPhone"><?php echo $_SESSION['last_generated']['data']['no_hp'] ?? 'Belum diisi'; ?></p>
                    </div>
                </div>
                
                <?php if (isset($_SESSION['last_generated'])): ?>
                <div style="margin-top: 20px; display: flex; gap: 15px;">
                    <a href="<?php echo $_SESSION['last_generated']['filename']; ?>" 
                       target="_blank" class="btn btn-secondary" style="flex: 1;">
                        <i class="fas fa-external-link-alt"></i> Buka File
                    </a>
                    <a href="download.php?file=<?php echo urlencode($_SESSION['last_generated']['filename']); ?>" 
                       class="btn btn-primary" style="flex: 1;">
                        <i class="fas fa-download"></i> Download
                    </a>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Daftar File yang Sudah Dibuat -->
        <div class="file-list">
            <h2 class="section-title"><i class="fas fa-folder-open"></i> Landing Pages Tersimpan</h2>
            
            <?php if (count($html_files) > 0): ?>
            <div class="file-grid">
                <?php foreach ($html_files as $file): 
                    $filetime = date('Y-m-d H:i:s', filemtime($file));
                    $filesize = round(filesize($file) / 1024, 2);
                ?>
                <div class="file-card">
                    <div class="file-name">
                        <i class="fas fa-file-code"></i> <?php echo $file; ?>
                    </div>
                    <div class="file-date">
                        <i class="far fa-clock"></i> <?php echo $filetime; ?>
                    </div>
                    <div class="file-actions">
                        <a href="<?php echo $file; ?>" target="_blank">
                            <i class="fas fa-eye"></i> View
                        </a>
                        <a href="download.php?file=<?php echo urlencode($file); ?>">
                            <i class="fas fa-download"></i> Download
                        </a>
                        <a href="#" onclick="deleteFile('<?php echo $file; ?>')" style="background: rgba(225, 29, 29, 0.2);">
                            <i class="fas fa-trash"></i> Delete
                        </a>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <div style="text-align: center; padding: 40px; color: #888;">
                <i class="fas fa-folder-open fa-3x" style="margin-bottom: 20px;"></i>
                <p>Belum ada landing page yang dibuat</p>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Real-time character counter
        const titleInput = document.getElementById('title');
        const descInput = document.getElementById('description');
        const titleCount = document.getElementById('titleCount');
        const descCount = document.getElementById('descCount');
        
        function updateCharCount(input, counter, max) {
            const length = input.value.length;
            counter.textContent = `${length}/${max}`;
            
            if (length > max * 0.8) {
                counter.classList.add('warning');
                counter.classList.remove('danger');
            } else if (length > max * 0.95) {
                counter.classList.remove('warning');
                counter.classList.add('danger');
            } else {
                counter.classList.remove('warning', 'danger');
            }
        }
        
        // Real-time preview update
        function updatePreview() {
            document.getElementById('previewTitle').textContent = titleInput.value || 'Belum diisi';
            document.getElementById('previewUrl').textContent = document.getElementById('url').value || 'Belum diisi';
            document.getElementById('previewDesc').textContent = descInput.value ? 
                (descInput.value.length > 100 ? descInput.value.substring(0, 100) + '...' : descInput.value) : 
                'Belum diisi';
            document.getElementById('previewImage').textContent = document.getElementById('url_gambar').value || 'Belum diisi';
            document.getElementById('previewPhone').textContent = document.getElementById('no_hp').value || 'Belum diisi';
        }
        
        // Event listeners
        titleInput.addEventListener('input', function() {
            updateCharCount(this, titleCount, 100);
            updatePreview();
        });
        
        descInput.addEventListener('input', function() {
            updateCharCount(this, descCount, 500);
            updatePreview();
        });
        
        document.getElementById('url').addEventListener('input', updatePreview);
        document.getElementById('url_gambar').addEventListener('input', updatePreview);
        document.getElementById('no_hp').addEventListener('input', updatePreview);
        
        // Initialize counters
        updateCharCount(titleInput, titleCount, 100);
        updateCharCount(descInput, descCount, 500);
        updatePreview();
        
        // Reset form function
        function resetForm() {
            if (confirm('Yakin ingin mereset semua data?')) {
                document.getElementById('landingForm').reset();
                updateCharCount(titleInput, titleCount, 100);
                updateCharCount(descInput, descCount, 500);
                updatePreview();
            }
        }
        
        // Delete file function
        function deleteFile(filename) {
            if (confirm(`Yakin ingin menghapus file "${filename}"?`)) {
                fetch('delete.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'file=' + encodeURIComponent(filename)
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('File berhasil dihapus');
                        location.reload();
                    } else {
                        alert('Error: ' + data.error);
                    }
                })
                .catch(error => {
                    alert('Error: ' + error);
                });
            }
        }
        
        // Auto-hide notifications after 5 seconds
        setTimeout(() => {
            const notifications = document.querySelectorAll('.notification');
            notifications.forEach(notification => {
                notification.style.animation = 'slideIn 0.3s ease reverse';
                setTimeout(() => notification.remove(), 300);
            });
        }, 5000);
    </script>
</body>
</html>